<?php
/**
 * Elementor Vehicle Seller Contact widget.
 *
 * @package cardealer-helper/elementor
 * @since   5.0.0
 */

namespace Cdhl_Elementor\Widgets;

use Elementor\Utils;
use Elementor\Widget_Base;
use Elementor\Controls_Manager;

use Cdhl_Elementor\Widget_Controller\Widget_Controller;
use Cdhl_Elementor\Group_Controls;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Elementor banner widget.
 *
 * Elementor widget that displays an banner.
 *
 * @since 5.0.0
 */
class Vehicle_Seller_Contact extends Widget_Controller {

	/**
	 * Widget slug
	 *
	 * @var string
	 */
	protected $widget_slug = 'vehicle-seller-contact';

	/**
	 * Widget icon
	 *
	 * @var string
	 */
	protected $widget_icon = 'cdhl-widget-icon';

	/**
	 * Widget keywords
	 *
	 * @var array
	 */
	protected $keywords = array( 'cars', 'vehicle', 'detail', 'single', 'seller', 'contact' );

	/**
	 * Retrieve the widget title.
	 *
	 * @since 5.0.0
	 *
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return esc_html__( 'Vehicle Seller Contact', 'cardealer-helper' );
	}

	/**
	 * Retrieve the list of categories the widget belongs to.
	 *
	 * Used to determine where to display the widget in the editor.
	 *
	 * Note that currently Elementor supports only one category.
	 * When multiple categories passed, Elementor uses the first one.
	 *
	 * @since 5.0.0
	 *
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return array( CDHL_ELEMENTOR_VEHIVLE_SINGLE_CAT );
	}

	/**
	 * Register the widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 5.0.0
	 *
	 * @access protected
	 */
	protected function register_controls() {
		$usertype_labels = $this->usertype_labels();

		$this->start_controls_section(
			'section_content_general',
			array(
				'label' => esc_html__( 'General', 'cardealer-helper' ),
				'tab'   => Controls_Manager::TAB_CONTENT,
			)
		);

		$this->add_control(
			'contact_form',
			[
				'label'       => esc_html__( 'Contact Form', 'cardealer-helper' ),
				'type'        => Controls_Manager::SELECT,
				'description' => esc_html__( 'Please select contact form.', 'cardealer-helper' ),
				'options'     => cdhl_get_cf7_options(),
			]
		);

		$this->add_control(
			'show_title',
			[
				'label'        => esc_html__( 'Show Title', 'cardealer-helper' ),
				'type'         => \Elementor\Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Show', 'cardealer-helper' ),
				'label_off'    => esc_html__( 'Hide', 'cardealer-helper' ),
				'return_value' => 'yes',
				'default'      => 'no',
			]
		);

		$this->add_control(
			'dealer_title',
			[
				'label'       => esc_html__( 'Dealer Title', 'cardealer-helper' ),
				'type'        => Controls_Manager::TEXT,
				'description' => esc_html__( 'Enter the title to display for users with the "Car Dealer" role.', 'cardealer-helper' ),
				'default'     => $usertype_labels['dealer'],
				'condition'   => [
					'show_title' => 'yes'
				],
			]
		);

		$this->add_control(
			'user_title',
			[
				'label'       => esc_html__( 'User Title', 'cardealer-helper' ),
				'type'        => Controls_Manager::TEXT,
				'description' => esc_html__( 'Enter the title to display for users other than the "Car Dealer" role.', 'cardealer-helper' ),
				'default'     => $usertype_labels['user'],
				'condition'   => [
					'show_title' => 'yes'
				],
			]
		);

		$this->end_controls_section();
	}

	public function usertype_labels() {
		$usertypes       = function_exists( 'cdfs_get_usertypes' ) ? cdfs_get_usertypes() : array();
		$usertype_labels = array();
		foreach ( $usertypes as $usertype => $usertype_data ) {
			$usertype_label = ( isset( $usertype_data['label'] ) && ! empty( $usertype_data['label'] ) ) ? $usertype_data['label'] : $usertype_data['label_original'];
			$usertype_labels[ $usertype ] = sprintf(
				esc_html__( 'Contact %s', 'cardealer-helper' ),
				$usertype_label
			);
		}
		return $usertype_labels;
	}
}
