<?php
/**
 * Plugin Name:     Affiliate Coupons (Pro)
 * Plugin URI:      https://affcoups.com
 * Description:     Extending the free plugin with more awesome features!
 * Version:         1.2.0
 * Author:          Affiliate Coupons
 * Author URI:      https://affcoups.com/
 * Text Domain:     affiliate-coupons-pro
 *
 * @author          fdmedia
 * @copyright       Copyright (c) fdmedia
 */

// Exit if accessed directly
if( ! defined( 'ABSPATH' ) ) exit;

if( ! class_exists( 'Affiliate_Coupons_Pro' ) ) :

	/**
	 * Main Affiliate_Coupons_Pro class
	 *
	 * @since       1.0.0
	 */
	final class Affiliate_Coupons_Pro {
		/** Singleton *************************************************************/

		/**
		 * Affiliate_Coupons_Pro instance.
		 *
		 * @access private
		 * @since  1.0.0
		 * @var    Affiliate_Coupons_Pro The one true Affiliate_Coupons_Pro
		 */
		private static $instance;

        /**
         * The settings instance variable.
         *
         * @access public
         * @since  1.0.0
         * @var    Affcoups_Pro_Settings
         */
        public $settings;

        /**
         * The stats instance variable.
         *
         * @access public
         * @since  1.0.0
         * @var    Affcoups_Pro_Stats
         */
        public $stats;

        /**
         * The stats db instance variable.
         *
         * @access public
         * @since  1.0.0
         * @var    Affcoups_Pro_Stats
         */
        public $stats_db;

		/**
		 * The version number of Affiliate_Coupons_Pro.
		 *
		 * @access private
         * @since  1.0.0
		 * @var    string
		 */
		private $version = '1.1.12';

		/**
		 * Main Affiliate_Coupons_Pro Instance
		 *
		 * Insures that only one instance of Affiliate_Coupons_Pro exists in memory at any one
		 * time. Also prevents needing to define globals all over the place.
		 *
		 * @since 1.0
		 * @static
		 * @staticvar array $instance
		 * @uses Affiliate_Coupons_Pro::setup_globals() Setup the globals needed
		 * @uses Affiliate_Coupons_Pro::includes() Include the required files
		 * @uses Affiliate_Coupons_Pro::setup_actions() Setup the hooks and actions
		 * @return Affiliate_Coupons_Pro
		 */
		public static function instance() {
			if ( ! isset( self::$instance ) && ! ( self::$instance instanceof Affiliate_Coupons_Pro ) ) {
				self::$instance = new Affiliate_Coupons_Pro;

                if ( ! function_exists( 'Affiliate_Coupons' ) ) {

                    add_action( 'admin_notices', array( 'Affiliate_Coupons_Pro', 'base_plugin_required_notice' ) );
                    add_filter( 'plugin_row_meta', array( 'Affiliate_Coupons_Pro', 'base_plugin_required_row_meta_notice' ), 10, 2 );

                    return self::$instance;

                } elseif ( function_exists( 'Affiliate_Coupons' ) && version_compare( PHP_VERSION, '5.3', '<' ) ) {
                    return self::$instance;
                }

				self::$instance->setup_constants();
				self::$instance->includes();
                self::$instance->setup_objects();
                self::$instance->load_textdomain();
			}

			return self::$instance;
		}

		/**
		 * Throw error on object clone
		 *
		 * The whole idea of the singleton design pattern is that there is a single
		 * object therefore, we don't want the object to be cloned.
		 *
		 * @since 1.0.0
		 * @access protected
		 * @return void
		 */
		public function __clone() {
			// Cloning instances of the class is forbidden
			_doing_it_wrong( __FUNCTION__, __( 'Cheatin&#8217; huh?', 'affiliate-coupons-pro' ), '1.0' );
		}

		/**
		 * Disable unserializing of the class
		 *
		 * @since 1.0.0
		 * @access protected
		 * @return void
		 */
		public function __wakeup() {
			// Unserializing instances of the class is forbidden
			_doing_it_wrong( __FUNCTION__, __( 'Cheatin&#8217; huh?', 'affiliate-coupons-pro' ), '1.0' );
		}

        /**
         * Show a warning in case the base plugin is not activated (admin notice)
         *
         * @static
         * @access private
         * @since 1.0.0
         * @return void
         */
        public static function base_plugin_required_notice() {
            ?>
            <div class="error">
                <p>
                    <?php printf( wp_kses( __( 'Please install the free version of <a href="%s">Affiliate Coupons</a> which is required to run the PRO version.', 'affiliate-coupons-pro' ), array(  'a' => array( 'href' => array() ) ) ), esc_url( admin_url( 'plugin-install.php?s=affiliate+coupons&tab=search&type=term' ) ) ); ?>
                </p>
            </div>
            <?php
        }

        /**
         * Show a warning in case the base plugin is not activated (plugin row meta)
         *
         * @param $input
         * @param $file
         * @return array
         */
        public static function base_plugin_required_row_meta_notice( $input, $file ) {

            if ( $file != 'affiliate-coupons-pro/affiliate-coupons-pro.php' )
                return $input;

            $links = array(
                '<p style="margin: .8em 0 0; color: red;">' . sprintf( wp_kses( __( 'Please install the free version of <a href="%s">Affiliate Coupons</a> which is required to run the PRO version.', 'affiliate-coupons-pro' ), array(  'a' => array( 'href' => array() ) ) ), esc_url( admin_url( 'plugin-install.php?s=affiliate+coupons&tab=search&type=term' ) ) ) . '</p>',
            );

            $input = array_merge( $input, $links );

            return $input;
        }

		/**
		 * Setup plugin constants
		 *
		 * @access private
		 * @since 1.0.0
		 * @return void
		 */
		private function setup_constants() {
			// Plugin version
			if ( ! defined( 'AFFCOUPS_PRO_VERSION' ) ) {
				define( 'AFFCOUPS_PRO_VERSION', $this->version );
			}

			// Plugin Folder Path
			if ( ! defined( 'AFFCOUPS_PRO_PLUGIN_DIR' ) ) {
				define( 'AFFCOUPS_PRO_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );
			}

			// Plugin Folder URL
			if ( ! defined( 'AFFCOUPS_PRO_PLUGIN_URL' ) ) {
				define( 'AFFCOUPS_PRO_PLUGIN_URL', plugin_dir_url( __FILE__ ) );
			}

			// Plugin Root File
			if ( ! defined( 'AFFCOUPS_PRO_PLUGIN_FILE' ) ) {
				define( 'AFFCOUPS_PRO_PLUGIN_FILE', __FILE__ );
			}

			// Docs URL
			if ( ! defined( 'AFFCOUPS_PRO_DOCS_URL' ) ) {
				define( 'AFFCOUPS_PRO_DOCS_URL', 'https://affcoups.com/support/knb/' );
			}

            // License Server URL
            if ( ! defined( 'AFFCOUPS_PRO_LICENSE_SERVER_URL' ) ) {
                define( 'AFFCOUPS_PRO_LICENSE_SERVER_URL', 'https://affcoups.com' );
            }

            // License Item ID
            if ( ! defined( 'AFFCOUPS_PRO_LICENSE_ITEM_ID' ) ) {
                define( 'AFFCOUPS_PRO_LICENSE_ITEM_ID', 28222 );
            }
		}

		/**
		 * Include required files
		 *
		 * @access private
		 * @since 1.0
		 * @return void
		 */
		private function includes() {

			// Core
			require_once AFFCOUPS_PRO_PLUGIN_DIR . 'includes/functions.php';
            require_once AFFCOUPS_PRO_PLUGIN_DIR . 'includes/class-stats-db.php';
            require_once AFFCOUPS_PRO_PLUGIN_DIR . 'includes/class-coupon-extended.php';
            require_once AFFCOUPS_PRO_PLUGIN_DIR . 'includes/stats-functions.php';
            require_once AFFCOUPS_PRO_PLUGIN_DIR . 'includes/search-filters-functions.php';
            require_once AFFCOUPS_PRO_PLUGIN_DIR . 'includes/ajax.php';
            require_once AFFCOUPS_PRO_PLUGIN_DIR . 'includes/hooks.php';
            require_once AFFCOUPS_PRO_PLUGIN_DIR . 'includes/code-reveal.php';
            require_once AFFCOUPS_PRO_PLUGIN_DIR . 'includes/meta-boxes.php';
            require_once AFFCOUPS_PRO_PLUGIN_DIR . 'includes/widgets.php';
            require_once AFFCOUPS_PRO_PLUGIN_DIR . 'includes/template-functions.php';

			// Other
            require_once AFFCOUPS_PRO_PLUGIN_DIR . 'includes/install.php';
			require_once AFFCOUPS_PRO_PLUGIN_DIR . 'includes/scripts.php';

			if ( is_admin() || ( defined( 'WP_CLI' ) && WP_CLI ) ) {
				// Bootstrap.
				require_once AFFCOUPS_PRO_PLUGIN_DIR . 'includes/admin/plugins.php';
                require_once AFFCOUPS_PRO_PLUGIN_DIR . 'includes/admin/manage-coupons.php';
                require_once AFFCOUPS_PRO_PLUGIN_DIR . 'includes/admin/ajax.php';

				// Classes
				require_once AFFCOUPS_PRO_PLUGIN_DIR . 'includes/admin/class-settings.php';
                require_once AFFCOUPS_PRO_PLUGIN_DIR . 'includes/admin/class-stats.php';

                // Updater
                require_once AFFCOUPS_PRO_PLUGIN_DIR . 'includes/admin/class-plugin-updater.php';

                // Licensing
                require_once AFFCOUPS_PRO_PLUGIN_DIR . 'includes/admin/class-license-handler.php';

				// Upgrades
				//require_once AFFCOUPS_PRO_PLUGIN_DIR . 'includes/admin/upgrades.php';

				// CSV uploader-downloader
				require_once AFFCOUPS_PRO_PLUGIN_DIR . 'includes/admin/upload-coupon.php';
				require_once AFFCOUPS_PRO_PLUGIN_DIR . 'includes/admin/export-coupon.php';
			}
		}

		/**
		 * Setup all objects
		 *
		 * @access public
		 * @since 1.6.2
		 * @return void
		 */
		public function setup_objects() {

            self::$instance->stats_db = new Affcoups_Pro_DB_Stats();

			if ( is_admin() || ( defined( 'WP_CLI' ) && WP_CLI ) ) {
				self::$instance->settings = new Affcoups_Pro_Settings();
                self::$instance->stats = new Affcoups_Pro_Stats();
			}

            self::$instance->updater();
		}

        /**
         * Plugin Updater
         *
         * @access private
         * @since 1.0
         * @return void
         */
        private function updater() {

            if ( ! is_admin() || ! class_exists( 'Affcoups_Pro_Plugin_Updater' ) ) {
                return;
            }

            $license_key = affcoups_get_option( 'license_key' );

            if ( ! empty( $license_key ) ) {

                $plugin_updater = new Affcoups_Pro_Plugin_Updater(AFFCOUPS_PRO_LICENSE_SERVER_URL, __FILE__, array(
                    'version' => AFFCOUPS_PRO_VERSION,
                    'license' => $license_key,
                    'item_id' => AFFCOUPS_PRO_LICENSE_ITEM_ID,
                    'author' => 'fdmedia',
                    'url' => home_url()
                ));
            }
        }

		/**
		 * Loads the plugin language files
		 *
		 * @access public
		 * @since 1.0
		 * @return void
		 */
		public function load_textdomain() {

			// Set filter for plugin's languages directory
			$lang_dir = dirname( plugin_basename( AFFCOUPS_PRO_PLUGIN_FILE ) ) . '/languages/';

			/**
			 * Filters the languages directory path to use for Affiliate_Coupons_Pro.
			 *
			 * @param string $lang_dir The languages directory path.
			 */
			$lang_dir = apply_filters( 'affiliate_coupons_pro_languages_directory', $lang_dir );

			// Traditional WordPress plugin locale filter

			global $wp_version;

			$get_locale = get_locale();

			if ( $wp_version >= 4.7 ) {
				$get_locale = get_user_locale();
			}

			/**
			 * Defines the plugin language locale used in Affiliate_Coupons_Pro.
			 *
			 * @var $get_locale The locale to use. Uses get_user_locale()` in WordPress 4.7 or greater,
			 *                  otherwise uses `get_locale()`.
			 */
			$locale = apply_filters( 'plugin_locale', $get_locale, 'affiliate-coupons-pro' );
			$mofile = sprintf( '%1$s-%2$s.mo', 'affiliate-coupons-pro', $locale );

			// Setup paths to current locale file
			$mofile_local  = $lang_dir . $mofile;
			$mofile_global = WP_LANG_DIR . '/affiliate-coupons-pro/' . $mofile;

			if ( file_exists( $mofile_global ) ) {
				// Look in global /wp-content/languages/affiliate-coupons-pro/ folder
				load_textdomain( 'affiliate-coupons-pro', $mofile_global );
			} elseif ( file_exists( $mofile_local ) ) {
				// Look in local /wp-content/plugins/affiliate-coupons-pro/languages/ folder
				load_textdomain( 'affiliate-coupons-pro', $mofile_local );
			} else {
				// Load the default language files
				load_plugin_textdomain( 'affiliate-coupons-pro', false, $lang_dir );
			}
		}
	}
endif; // End if class_exists check

/**
 * The main function responsible for returning the one true Affiliate_Coupons_Pro
 * Instance to functions everywhere.
 *
 * Use this function like you would a global variable, except without needing
 * to declare the global.
 *
 * Example: <?php $Affiliate_Coupons_Pro = Affiliate_Coupons_Pro(); ?>
 *
 * @since 1.0
 * @return Affiliate_Coupons_Pro The one true Affiliate_Coupons_Pro Instance
 */
function Affiliate_Coupons_Pro() {
	return Affiliate_Coupons_Pro::instance();
}

/**
 * Init plugin
 *
 * @return bool
 */
function affcoups_pro_load() {
    Affiliate_Coupons_Pro();

    return true;
}
add_action( 'plugins_loaded', 'affcoups_pro_load', 20 );