<?php
/**
 * Settings Class
 *
 * @since       1.0.0
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ! class_exists( 'Affcoups_Pro_Settings' ) ) {

	class Affcoups_Pro_Settings {

		public $options;

		/**
		 * Affcoups_Pro_Settings constructor.
		 */
		public function __construct() {

			// Options
			$this->options = affcoups_get_options();

			// Hooks
            add_action( 'affcoups_settings_section_quickstart_available_features', array( &$this, 'section_quickstart_available_features' ) );
            add_action( 'affcoups_settings_section_quickstart_render',            array( &$this, 'section_quickstart_render' ) );
            add_filter( 'affcoups_settings',                                      array( &$this, 'register_licensing_settings' ) );
            add_filter( 'affcoups_settings',                                      array( &$this, 'register_tracking_settings' ) );
            add_filter( 'affcoups_settings',                                      array( &$this, 'register_coupons_settings' ) );
            add_filter( 'affcoups_settings_code_options',                         array( &$this, 'add_code_reveal_option' ) );
            add_filter( 'affcoups_settings_click_to_reveal_button_text',          array( &$this, 'click_to_reveal_button_text' ) );
            add_action( 'affcoups_settings_styles_render',                        array( &$this, 'styles_render' ) );
            add_action( 'affcoups_settings_search_filters_render',                array( &$this, 'search_filters_render' ) );
            add_action( 'affcoups_settings_pagination_render',                    array( &$this, 'pagination_render' ) );
            add_filter( 'affcoups_settings_validate_input',                       array( &$this, 'validate_input' ) );
            add_filter( 'affcoups_settings_description_hide_options',             array( &$this, 'description_hide_options' ) ); //deepak 23jun23

		}

        /**
         * Validate input
         *
         * @param $input
         * @return mixed
         */
        function validate_input( $input ) {

            // License field
            $License_Handler = new Affcoups_Pro_License_Handler();

            $new_license_key = ( ! empty( $input['license_key'] ) ) ? $input['license_key'] : null;
            $old_license_key = ( isset( $this->options['license_key'] ) ) ? $this->options['license_key'] : null;

            // Validate license key update
            $license_data = $License_Handler->handle_update( $new_license_key, $old_license_key );

            if ( isset( $license_data->license ) ) {
                $input['license_status'] = $license_data->license;
                $input['license_error'] =  ( isset( $license_data->error ) ) ? $license_data->error : '';
                //affcoups_addlog( 'License key changed > Status: ' . $input['license_status'] . '(' . $input['license_error'] . ')' );
            }

            if ( ! $license_data && isset( $input['license_check'] ) && ! empty( $input['license_key'] ) ) {
                $license_data = $License_Handler->check_license( $input['license_key'] );

                if ( isset( $license_data->license ) ) {
                    $input['license_status'] = $license_data->license;
                    $input['license_error'] =  ( isset( $license_data->error ) ) ? $license_data->error : '';
                    //affcoups_addlog( 'License check > Result: ' . $input['license_status'] . '(' . $input['license_error'] . ')' );
                }
            }

            // Validate pagination
            // Set 'pagination' fallback value (instead of WP native totally removing) if customer unchecks the previously checked box
            //if ( empty( $input['pagination'] ) && isset( $this->options['pagination'] ) )
            //    $input['pagination'] = '0';

            //affcoups_debug( $input );

            return $input;
        }

        /**
         * Register licensing settings
         *
         * @param $settings
         * @return array
         */
		function register_licensing_settings( $settings ) {

		    $settings_licensing = apply_filters( 'affcoups_settings_licensing',
                array(
                    'icon' => 'admin-network',
                    'title' => __( 'Licensing', 'affiliate-coupons-pro' ),
                    'callback' => array( &$this, 'section_licensing_render' ),
                    'fields' => array(
                        'license_status' => array(
                            'title' => __( 'License Status', 'affiliate-coupons-pro' ),
                            'callback' => array( &$this, 'license_status_render' )
                        ),
                        'license_key' => array(
                            'title' => __( 'License Key', 'affiliate-coupons-pro' ),
                            'callback' => array( &$this, 'license_key_render' )
                        )
                    )
                )
            );

            $settings = affcoups_array_insert_after( $settings, 'quickstart', 'licensing', $settings_licensing );

		    return $settings;
        }

        /**
         * Register tracking settings
         *
         * @param $settings
         * @return array
         */
        function register_tracking_settings( $settings ) {

            $settings_tracking = apply_filters( 'affcoups_settings_tracking',
                array(
                    'icon' => 'chart-line',
                    'title' => __( 'Tracking & Statistics', 'affiliate-coupons-pro' ),
                    'fields' => array(
                        'stats' => array(
                            'title' => __( 'Statistics', 'affiliate-coupons-pro' ),
                            'callback' => array( &$this, 'stats_render' )
                        ),
                        'event_tracking' => array(
                            'title' => __( 'Event Tracking', 'affiliate-coupons-pro' ),
                            'callback' => array( &$this, 'event_tracking_render' )
                        )
                    )
                )
            );

            $settings = affcoups_array_insert_after( $settings, 'general', 'tracking', $settings_tracking );

            return $settings;
        }

        /**
         * Register coupons settings
         *
         * @param $settings
         * @return array
         */
        function register_coupons_settings( $settings ) {

            $settings_coupons = apply_filters( 'affcoups_settings_coupons',
                array(
                    'icon' => 'tag',
                    'title' => __( 'Coupon Pages', 'affiliate-coupons-pro' ),
                    'callback' => array( &$this, 'section_coupons_render' ),
                    'fields' => array(
                        'coupon_pages' => array(
                            'title' => __( 'Detail Pages', 'affiliate-coupons-pro' ),
                            'callback' => array( &$this, 'coupon_pages_render' )
                        ),
                        'coupon_redirect_to_pages' => array(
                            'title' => __( 'Redirect to Pages', 'affiliate-coupons-pro' ),
                            'callback' => array( &$this, 'coupon_redirect_to_pages_render' )
                        ),
                        'coupon_slug' => array(
                            'title' => __( 'Slug', 'affiliate-coupons-pro' ),
                            'callback' => array( &$this, 'coupon_slug_render' )
                        ),
                        'coupon_include_in_search' => array(
                            'title' => __( 'Search Results', 'affiliate-coupons-pro' ),
                            'callback' => array( &$this, 'coupon_include_in_search_render' )
                        )
                    )
                )
            );

            $settings = affcoups_array_insert_after( $settings, 'tracking', 'coupons', $settings_coupons );

            return $settings;
        }

        /**
         * Extend quickstart available features list
         */
        function section_quickstart_available_features() {

            ?>
            <li><a href="#styles"><?php esc_html_e( 'Styles', 'affiliate-coupons-pro' ); ?></a></li>
            <?php
        }

        /**
         * Extend quickstart section
         */
        function section_quickstart_render() {

            ?>
            <p id="styles" style="padding-top: 2.4em">
                <strong><?php esc_html_e( 'Styles', 'affiliate-coupons-pro' ); ?></strong><br/>
                <?php esc_html_e( 'You can easily select a style on a shortcode basis as follows:', 'affiliate-coupons-pro' ); ?>
            </p>
            <p>
                <code>[affcoups style="standard"]</code>
            </p>
            <p>
                <?php esc_html_e( 'The following styles are available right now:', 'affiliate-coupons-pro' ); ?>
                <code>standard</code>, <code>light</code> & <code>dark</code>
            </p>
            <?php
        }

        /**
         * Section licensing render
         */
        function section_licensing_render() {

            ?>
            <p><?php _e('In order to make use of our pro version and receive regularly updates, please enter your license key.', 'affiliate-coupons-pro' ); ?></p>
            <?php
        }

        /**
         * License status
         */
        function license_status_render() {

            $license_status = ( isset ( $this->options['license_status'] ) ) ? $this->options['license_status'] : '';
            $license_error = ( isset ( $this->options['license_error'] ) ) ? $this->options['license_error'] : '';

            switch ( $license_status) {
                case 'valid':
                    $license_status_color = 'green'; $license_status_label = __( 'Valid', 'affiliate-coupons-pro' );
                    break;
                case 'expired':
                    $license_status_color = 'orange'; $license_status_label = __( 'Expired', 'affiliate-coupons-pro' );
                    break;
                case 'invalid':
                    $license_status_color = 'red'; $license_status_label = __( 'Invalid', 'affiliate-coupons-pro' );
                    break;
                case 'failed':
                    $license_status_color = 'red'; $license_status_label = __( 'Failed.', 'affiliate-coupons-pro' );
                    break;
                default:
                    $license_status_color = 'gray'; $license_status_label = __( 'Disconnected', 'affiliate-coupons-pro' );
            }
            ?>
            <span style="color: <?php echo $license_status_color; ?>; font-weight: bold;"><?php echo $license_status_label; ?></span>

            <?php if ( ! empty( $license_error ) ) { ?>
                <code><?php echo esc_html( $license_error ); ?></code>
            <?php
            }
        }

        /**
         * License key
         */
        function license_key_render() {

            $license_key = ( isset ( $this->options['license_key'] ) ) ? $this->options['license_key'] : '';
            ?>
            <input id="affcoups_license_key" class="regular-text" name="affcoups_settings[license_key]" type="text" value="<?php echo esc_html( $license_key ); ?>" />

            <?php if ( ! empty( $license_key ) ) { ?>
                <input type="hidden" id="affcoups-pro-check-license" name="affcoups_settings[license_check]" value="0" />
                <?php submit_button( __('Check License', 'affiliate-coupons-pro' ), 'delete button-secondary', 'affcoups-pro-check-license-submit', false, array( 'style' => 'vertical-align: 3%;') ); ?>
            <?php } ?>
            <?php
        }

        /**
         * Section coupon settings
         */
        function section_coupons_render() {

            ?>
            <p><?php _e('Here you find more settings in case you want to make use of coupon posts.', 'affiliate-coupons-pro' ); ?></p>
            <?php
        }

        /**
         * Render coupon detail pages settings
         */
        function coupon_pages_render() {

            $detail_pages = ( isset( $this->options['coupon_pages'] ) && $this->options['coupon_pages'] == '1' ) ? 1 : 0;
            ?>
            <p>
                <input type="checkbox" id="affcoups_coupon_pages" name="affcoups_settings[coupon_pages]" value="1" <?php echo( $detail_pages == 1 ? 'checked' : '' ); ?> />
                <label for="affcoups_coupon_pages"><?php esc_html_e( 'Activate in order to create pages for every single coupon', 'affiliate-coupons-pro' ); ?></label>
            </p>
            <?php
        }

        /**
         * Render coupon redirect to pages settings
         */
        function coupon_redirect_to_pages_render() {

            $coupon_redirect_to_pages = ( isset( $this->options['coupon_redirect_to_pages'] ) && $this->options['coupon_redirect_to_pages'] == '1' ) ? 1 : 0;
            ?>
            <p>
                <input type="checkbox" id="affcoups_coupon_redirect_to_pages" name="affcoups_settings[coupon_redirect_to_pages]" value="1" <?php echo( $coupon_redirect_to_pages == 1 ? 'checked' : '' ); ?> />
                <label for="affcoups_coupon_redirect_to_pages"><?php esc_html_e( 'Activate in order to redirect links and buttons to coupon pages first', 'affiliate-coupons-pro' ); ?></label>
            </p>
            <?php
        }

        /**
         * Render coupon slug settings
         */
        function coupon_slug_render() {

            $slug = ( ! empty( $this->options['coupon_slug'] ) ) ? esc_attr( trim( $this->options['coupon_slug'] ) ) : 'coupon';
            ?>
            <p>
                <input type="text" name="affcoups_settings[coupon_slug]" id="affcoups_coupon_slug" value="<?php echo esc_attr( trim( $slug ) ); ?>"/>
            </p>
            <p class="description"><?php _e('Defines the url slug for single coupon pages.', 'affiliate-coupons-pro' ); ?></p>
            <p>
                <small>
                    <?php
                    printf( wp_kses( __( 'After changing the coupon slug please <a href="%s">update your permalinks</a>.', 'affiliate-coupons-pro' ), array(  'a' => array( 'href' => array() ) ) ), esc_url( admin_url( 'options-permalink.php' ) ) );
                    ?>
                </small>
            </p>
            <?php
        }

        /**
         * Render coupon include in search settings
         */
        function coupon_include_in_search_render() {

            $include_in_search = ( isset( $this->options['coupon_include_in_search'] ) && $this->options['coupon_include_in_search'] == '1' ) ? 1 : 0;
            ?>
            <p>
                <input type="checkbox" id="affcoups_coupon_include_in_search" name="affcoups_settings[coupon_include_in_search]" value="1" <?php echo( $include_in_search == 1 ? 'checked' : '' ); ?> />
                <label for="affcoups_coupon_include_in_search"><?php esc_html_e( 'Activate in order include coupons in the front end search results', 'affiliate-coupons-pro' ); ?></label>
            </p>
            <?php
        }

        /**
         * Add code reveal setting option
         *
         * @param $code_options
         * @return mixed
         */
        function add_code_reveal_option( $code_options ) {

            $code_options['reveal'] = __( 'Click to reveal', 'affiliate-coupons-pro' );

            return $code_options;
        }

        /**
         * Let admins to override "Click to Reveal" button text
         */
        function click_to_reveal_button_text() {

            $button_text = ( ! empty( $this->options['click_to_reveal_button_text'] ) ) ? esc_attr( trim( $this->options['click_to_reveal_button_text'] ) ) : __( 'Show Code', 'affiliate-coupons-pro' );
            ?>
            <h4><?php esc_html_e( 'Click to Reveal: Button Text', 'affiliate-coupons-pro' ); ?></h4>
            <p>
                <input type="text" class="affcoups-input-click-to-reveal" name="affcoups_settings[click_to_reveal_button_text]" value="<?php echo esc_attr( $button_text ); ?>"/>
            </p>
            <?php

            /* Modal Button text Edit option*/
             $modal_button_text = ( ! empty( $this->options['modal_popup_button_text'] ) ) ? esc_attr( trim( $this->options['modal_popup_button_text'] ) ) : __( 'Go to Offer', 'affiliate-coupons-pro' );
             ?>
             <h4><?php esc_html_e( 'Popup coupon : Button Text', 'affiliate-coupons-pro' ); ?></h4>
             <p>
                 <input type="text" class="affcoups-input-social-share-msg" name="affcoups_settings[modal_popup_button_text]" value="<?php echo esc_attr( $modal_button_text ); ?>"/>
             </p>
             <?php
        }

        /**
         * Additional style settings
         */
        function styles_render() {

            $highlighted_border_color = ( isset( $this->options['highlighted_border_color'] ) ) ? $this->options['highlighted_border_color'] : '';
            ?>
            <h4><?php esc_html_e( 'Highlighted: Border Color', 'affiliate-coupons-pro' ); ?></h4>
            <p>
                <input type="text" class="affcoups-input-colorpicker" name="affcoups_settings[highlighted_border_color]" value="<?php echo esc_attr( $highlighted_border_color ); ?>"/>
            </p>
            <?php
        }
         /**
         * Render Coupon display field settings
         */
        function coupon_display_field_render() {

            $search_filters = ( isset( $this->options['search_filters'] ) && $this->options['search_filters'] == '1' ) ? 1 : 0;
            $hide_search_bar = ( isset( $this->options['hide_search_bar'] ) && $this->options['hide_search_bar'] == '1' ) ? 1 : 0;
            $hide_search_types = ( isset( $this->options['hide_search_types'] ) && $this->options['hide_search_types'] == '1' ) ? 1 : 0;
            $hide_search_category = ( isset( $this->options['hide_search_category'] ) && $this->options['hide_search_category'] == '1' ) ? 1 : 0;
            $hide_search_vendor = ( isset( $this->options['hide_search_vendor'] ) && $this->options['hide_search_vendor'] == '1' ) ? 1 : 0;
            ?>
            <p>
                <input type="checkbox" id="affcoups_search_filters" name="affcoups_settings[search_filters]" value="1" <?php echo( $search_filters == 1 ? 'checked' : '' ); ?> />
                <label for="affcoups_search_filters"><?php esc_html_e('Activate coupon search filters', 'affiliate-coupons-pro' ); ?></label>
            </p>
            <p>
                <input type="checkbox" id="affcoups_hide_search_bar" name="affcoups_settings[hide_search_bar]" value="1" <?php echo( $hide_search_bar == 1 ? 'checked' : '' ); ?> />
                <label for="affcoups_hide_search_bar"><?php esc_html_e( 'Hide coupon search bar on the front end', 'affiliate-coupons-pro' ); ?></label>
            </p>
            <p>
                <input type="checkbox" id="affcoups_hide_search_types" name="affcoups_settings[hide_search_types]" value="1" <?php echo( $hide_search_types == 1 ? 'checked' : '' ); ?> />
                <label for="affcoups_hide_search_types"><?php esc_html_e( 'Hide coupon types dropdown on the front end', 'affiliate-coupons-pro' ); ?></label>
            </p>
            <p>
                <input type="checkbox" id="affcoups_hide_search_category" name="affcoups_settings[hide_search_category]" value="1" <?php echo( $hide_search_category == 1 ? 'checked' : '' ); ?> />
                <label for="affcoups_hide_search_category"><?php esc_html_e( 'Hide coupon category dropdown on the front end', 'affiliate-coupons-pro' ); ?></label>
            </p>
            <p>
                <input type="checkbox" id="affcoups_hide_search_vendor" name="affcoups_settings[hide_search_vendor]" value="1" <?php echo( $hide_search_vendor == 1 ? 'checked' : '' ); ?> />
                <label for="affcoups_hide_search_vendor"><?php esc_html_e( 'Hide coupon vendor dropdown on the front end', 'affiliate-coupons-pro' ); ?></label>
            </p>
            <?php
        }

        /**
         * Render Coupon Search Filters settings
         */
        function search_filters_render() {

            $search_filters = ( isset( $this->options['search_filters'] ) && $this->options['search_filters'] == '1' ) ? 1 : 0;
            $hide_search_bar = ( isset( $this->options['hide_search_bar'] ) && $this->options['hide_search_bar'] == '1' ) ? 1 : 0;
            $hide_search_types = ( isset( $this->options['hide_search_types'] ) && $this->options['hide_search_types'] == '1' ) ? 1 : 0;
            $hide_search_category = ( isset( $this->options['hide_search_category'] ) && $this->options['hide_search_category'] == '1' ) ? 1 : 0;
            $hide_search_vendor = ( isset( $this->options['hide_search_vendor'] ) && $this->options['hide_search_vendor'] == '1' ) ? 1 : 0;
            ?>
            <p>
                <input type="checkbox" id="affcoups_search_filters" name="affcoups_settings[search_filters]" value="1" <?php echo( $search_filters == 1 ? 'checked' : '' ); ?> />
                <label for="affcoups_search_filters"><?php esc_html_e('Activate coupon search filters', 'affiliate-coupons-pro' ); ?></label>
            </p>
            <p>
                <input type="checkbox" id="affcoups_hide_search_bar" name="affcoups_settings[hide_search_bar]" value="1" <?php echo( $hide_search_bar == 1 ? 'checked' : '' ); ?> />
                <label for="affcoups_hide_search_bar"><?php esc_html_e( 'Hide coupon search bar on the front end', 'affiliate-coupons-pro' ); ?></label>
            </p>
            <p>
                <input type="checkbox" id="affcoups_hide_search_types" name="affcoups_settings[hide_search_types]" value="1" <?php echo( $hide_search_types == 1 ? 'checked' : '' ); ?> />
                <label for="affcoups_hide_search_types"><?php esc_html_e( 'Hide coupon types dropdown on the front end', 'affiliate-coupons-pro' ); ?></label>
            </p>
            <p>
                <input type="checkbox" id="affcoups_hide_search_category" name="affcoups_settings[hide_search_category]" value="1" <?php echo( $hide_search_category == 1 ? 'checked' : '' ); ?> />
                <label for="affcoups_hide_search_category"><?php esc_html_e( 'Hide coupon category dropdown on the front end', 'affiliate-coupons-pro' ); ?></label>
            </p>
            <p>
                <input type="checkbox" id="affcoups_hide_search_vendor" name="affcoups_settings[hide_search_vendor]" value="1" <?php echo( $hide_search_vendor == 1 ? 'checked' : '' ); ?> />
                <label for="affcoups_hide_search_vendor"><?php esc_html_e( 'Hide coupon vendor dropdown on the front end', 'affiliate-coupons-pro' ); ?></label>
            </p>
            <?php
        }

//        /**
//         * Set the Pagination default setting
//         */
//        private function maybe_set_pagination_defaults() {
//
//            $update = false;
//
//            // Pagination is active by default
//            if ( ! isset( $this->options['pagination'] ) ) {
//
//                $this->options['pagination'] = '1';
//                $update = true;
//            }
//
//            if ( ! isset( $this->options['pagination_per_page'] ) ) {
//
//                $this->options['pagination_per_page'] = 10;
//                $update = true;
//            }
//
//            if ( ! $update )
//                return;
//
//            update_option( 'affcoups_settings', $this->options );
//        }

        /**
         * Render Pagination settings
         */
        function pagination_render() {

//            $this->maybe_set_pagination_defaults();

            $pagination = ( isset( $this->options['pagination'] ) && $this->options['pagination'] == '1' ) ? 1 : 0;

            if ( ! empty( $this->options['pagination_per_page'] )
                && is_numeric( $this->options['pagination_per_page'] )
                && intval( $this->options['pagination_per_page'] ) > 0
            ){
                $pagination_per_page = intval( $this->options['pagination_per_page'] );
            } else {
                $pagination_per_page = 10;
            }

            $pagination_button_text = ( ! empty( $this->options['pagination_button_text'] ) ) ? esc_html( $this->options['pagination_button_text'] ) : esc_html__( 'Load More', 'affiliate-coupons-pro' );
            ?>
            <p>
                <input type="checkbox" id="affcoups_pagination" name="affcoups_settings[pagination]" value="1" <?php echo( $pagination == 1 ? 'checked' : '' ); ?> />
                <label for="affcoups_pagination"><?php esc_html_e( 'Activate coupons pagination', 'affiliate-coupons-pro' ); ?></label>
            </p>

            <h4><?php esc_html_e( 'Coupons per page', 'affiliate-coupons-pro' ); ?></h4>
            <p>
                <input type="number" id="affcoups_pagination_per_page" name="affcoups_settings[pagination_per_page]" min="1" value="<?php echo esc_attr( trim( $pagination_per_page ) ); ?>" />
            </p>

            <p class="desc">
                <?php esc_html_e( 'The default number of coupons per page.', 'affiliate-coupons-pro' ); ?>
            </p>

            <h4><?php esc_html_e( 'Pagination Button Text', 'affiliate-coupons-pro' ); ?></h4>
            <p>
                <input type="text" id="affcoups_pagination_button_text" name="affcoups_settings[pagination_button_text]" value="<?php echo $pagination_button_text; ?>" />
            </p>
            <?php
        }

        /**
         * Render Stats Settings
         */
        function stats_render() {

            $tracking_disabled = ( isset( $this->options['tracking_disabled'] ) && $this->options['tracking_disabled'] == '1' ) ? 1 : 0;
            ?>
            <p>
                <input type="checkbox" id="affcoups_tracking_disabled" name="affcoups_settings[tracking_disabled]" value="1" <?php echo( $tracking_disabled == 1 ? 'checked' : '' ); ?> />
                <label for="affcoups_tracking_disabled"><?php esc_html_e( 'Disable built-in tracking and statistics', 'affiliate-coupons-pro' ); ?></label>
            </p>
            <p class="description">
                <?php _e('Please only activate this checkbox if you experience issues or do not need any statistics at all.', 'affiliate-coupons-pro'); ?>
            </p>
            <?php
        }

        /**
         * Render Event Tracking Settings
         */
        function event_tracking_render() {

            $event_tracking = ( isset( $this->options['event_tracking'] ) && $this->options['event_tracking'] == '1' ) ? 1 : 0;
            ?>
            <p>
                <input type="checkbox" id="affcoups_event_tracking" name="affcoups_settings[event_tracking]" value="1" <?php echo( $event_tracking == 1 ? 'checked' : '' ); ?> />
                <label for="affcoups_event_tracking"><?php esc_html_e( 'Activate in order to track clicks on links by creating events via your favorite tracking tool', 'affiliate-coupons-pro' ); ?></label>
            </p>
            <p class="description">
                <?php _e('Currently supported:', 'affiliate-coupons-pro'); ?> Google Analytics, Google Tag Manager & Matomo (Piwik).
            </p>
            <?php
        }
	}
}