<?php
/**
 * Stats Database Class
 *
 * This file contains class for the stats DB table handling.
 *
 * @since     1.0.0
 */

// Quit if accessed outside WP context.
defined( 'ABSPATH' ) || exit;

/**
 * Stats Database Table
 *
 * @since 1.0.0
 */
class Affcoups_Pro_DB_Stats {

    /**
     * Database table name.
     *
     * @access public
     * @var    string
     */
    public $table_name;

    /**
     * Database version.
     *
     * @access public
     * @var    string
     */
    public $version;

    /**
	 * Definition of the custom table.
	 *
	 * @since   1.0.0
	 */
    public function __construct() {

		// Global.
		global $wpdb;

		// Name.
        $table = 'affcoups_stats';
        $this->table_name  = $wpdb->get_blog_prefix() . $table;

		// As array.
		$wpdb->tables[] = $table;

		// With prefix.
		$wpdb->$table = $this->table_name;
	}

    /**
     * Adds an entry to the database.
     *
     * @access public
     *
     * @param array $data Optional. Arguments for adding a new entry. Default empty array.
     * @return int ID of the added entry.
     */
    public function add( $data = array() ) {

        if ( empty( $data['coupon_id'] ) || empty( $data['type'] ) )
            return false;

        /* Global vars */
        global $wpdb;

        $defaults = array(
            'coupon_id' => 0,
            'post_id' => 0,
            'type' => '',
            'created_at' => date( 'Y-m-d H:i:s', time() )
        );

        /**
         * Parse incoming $args into an array and merge it with $defaults
         */
        $data = wp_parse_args( $data, $defaults );

        // Insert.
        $inserted = $wpdb->insert( $this->table_name, $data );

        return $inserted;
    }

    public function get_popular_coupons( $args = array() ) {
        global $wpdb;

        $defaults = array(
            'number'       => 10,
            'offset'       => 0,
            'type'         => 'click',
            'order'        => 'DESC',
            'orderby'      => 'amount',
        );

        $args = wp_parse_args( $args, $defaults );

        // Number
        if ( $args['number'] < 1 )
            $args['number'] = 999999999999;

        // Types
        if ( ! in_array( $args['type'], array( 'view', 'copy', 'click' ) ) )
            $args['type'] = 'click';

        // Query database
        $results = $wpdb->get_results(
            $wpdb->prepare(
                "SELECT coupon_id, count(*) AS amount FROM {$this->table_name} WHERE `type` = '{$args['type']}' GROUP BY coupon_id ORDER BY {$args['orderby']} {$args['order']} LIMIT %d, %d;",
                absint( $args['offset'] ),
                absint( $args['number'] )
            )
        );

        return $results;
    }

    /**
     * Returns the number of results found for a given query
     *
     * @param $coupon_id
     * @param $type
     * @return int
     */
    public function count( $coupon_id, $type ) {

        if ( empty( $coupon_id ) || empty( $type ) )
            return 0;

        /* Global vars */
        global $wpdb;

        $coupon_ids = esc_sql( intval( $coupon_id ) );
        $type = esc_sql( $type );

        $results = $wpdb->get_var(
            "SELECT COUNT(id) FROM {$this->table_name} WHERE `coupon_id` IN( {$coupon_ids} ) AND `type` LIKE '$type';" // "SELECT COUNT(id) FROM {$this->table_name} WHERE `coupon_id` IN( {$coupon_ids} ) AND `type` LIKE {$type};"
        );

        return $results;
    }

    /**
     * Reset all entries for a specific coupon id
     *
     * @param $coupon_id
     */
    public function reset( $coupon_id ) {

        if ( empty( $coupon_id ) )
            return;

        /* Global vars */
        global $wpdb;

        $coupon_ids = esc_sql( intval( $coupon_id ) );

        $wpdb->query( "DELETE FROM {$this->table_name} WHERE `coupon_id` IN( {$coupon_ids} )" );
    }

	/**
	 * Create the table.
	 *
	 * @since   1.0.0
	 */
	public function create() {

		global $wpdb;

		// Include.
		require_once( ABSPATH . 'wp-admin/includes/upgrade.php' );

		// Create.
        $sql = "CREATE TABLE {$this->table_name} (
			id bigint(20) NOT NULL AUTO_INCREMENT,
			type tinytext NOT NULL,
			coupon_id bigint(20) NOT NULL,
			post_id bigint(20) NOT NULL,
			created_at datetime NOT NULL,
			PRIMARY KEY  (id),
			KEY coupon_id (coupon_id),
			KEY post_id (post_id),
			KEY created_at (created_at)
			)";

        dbDelta( $sql );

        update_option( $this->table_name . '_db_version', $this->version );
	}

	/**
	 * Remove the custom table.
	 *
	 * @since   1.0.0
	 */
	public function drop() {

		global $wpdb;

		// Remove.
		$wpdb->query( "DROP TABLE IF EXISTS {$this->table_name}" );
	}
}
