<?php
/**
 * Hooks
 *
 * @since       1.0.0
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Load custom template
 *
 * @param $template_file
 * @param $template
 * @param $type
 * @return string
 */
function affcoups_pro_get_template_file( $template_file, $template, $type ) {

    // Check theme folder
    if ( $custom_template_file = locate_template( array( 'affcoups/' . $template . '.php' ) ) ) {
        return $custom_template_file;
    }

    if ( in_array( $template, array( 'banner', 'widget-small' ) ) ) {
        return AFFCOUPS_PRO_PLUGIN_DIR . 'templates/' . $template . '.php';
    }

    return $template_file;
}
add_filter( 'affcoups_template_file', 'affcoups_pro_get_template_file', 10, 3 );

/**
 * Extend widget template options
 *
 * @param $options
 * @return mixed
 */
function affcoups_pro_template_options( $options ) {

    $options['banner'] = __( 'Banner', 'affiliate-coupons-pro' );

    return $options;
}
add_filter( 'affcoups_template_options', 'affcoups_pro_template_options', 10 );

/**
 * Extend widget template options
 *
 * @param $options
 * @return mixed
 */
function affcoups_pro_widget_template_options( $options ) {

    $options['widget-small'] = __( 'Small', 'affiliate-coupons-pro' );

    return $options;
}
add_filter( 'affcoups_widget_template_options', 'affcoups_pro_widget_template_options', 10 );

/**
 * Apply Coupon Search Filters on frontend
 *
 * @param   array $coupons
 * @param   array $args
 *
 * @return  array
 */
function affcoups_pro_search_filters( $coupons, $args ) {

    //affcoups_debug_log( '' );
    //affcoups_debug_log( '------- SEARCH FILTERS START -------' );

    $coupon_ids = array();

    foreach ( $coupons as $coupon ) {
        $coupon_ids[] = $coupon->id;
    }
    // Default
    $args['sorted_coupon_ids'] = $coupon_ids;

    if ( ! isset( $args['search_filters'] ) || true !== $args['search_filters'] ) {
        return array( 'coupons' => $coupons, 'args' => $args );
    }

    // Coupon ids for $args['post__in']
    $coupon_ids = array();

    foreach ( $coupons as $coupon ) {
        $coupon_ids[] = $coupon->id;
    }

    if ( empty( $coupon_ids ) ) {
        return array( 'coupons' => $coupons, 'args' => $args );
    }

    $args['post__in'] = $coupon_ids;
    $args['posts_per_page'] = sizeof( $coupon_ids );

    $coupons = affcoups_get_coupons( $args );

    // Sotred coupon ids
    $sorted_coupon_ids = array();

    foreach ( $coupons as $coupon ) {
        $sorted_coupon_ids[] = $coupon->id;
    }

    $args['sorted_coupon_ids'] = $sorted_coupon_ids;
    $args['paginated_coupon_ids'] = $sorted_coupon_ids;

    //affcoups_debug_log( '' );
    //affcoups_debug_log( '------- SEARCH FILTERS END -------' );
    //affcoups_debug_log( '-----------------------------------------' );

    return array( 'coupons' => $coupons, 'args' => $args );
}
add_filter( 'affcoups_coupons_search_filters', 'affcoups_pro_search_filters', 10, 2 );

/**
 * Include Coupon Search Filters template
 *
 * @param array $coupons
 */
function affcoups_pro_search_filters_frontend( $coupons, $args ) {

    if ( ! isset( $args['search_filters'] ) || true !== $args['search_filters'] ) {
        return;
    }

    affcoups_pro_tpl_the_search_filters( $args );
}
add_action( 'affcoups_template_begin', 'affcoups_pro_search_filters_frontend', 10, 2 );

/**
 * Apply Coupon Pagination on frontend
 *
 * @param $coupons
 * @param $args
 *
 * @return array
 */
function affcoups_pro_pagination( $coupons, $args ) {

    //affcoups_debug_log( '' );
    //affcoups_debug_log( '------- PAGINATION START -------' );

    // if shortcode has no 'pagination' attribute
    if ( ! isset( $args['pagination'] ) || true !== $args['pagination'] ) {
        return array( 'coupons' => $coupons, 'args' => $args );
    }

    if ( ! isset( $args['pagination_per_page'] ) || intval( $args['pagination_per_page'] ) < 1 ) {
        return array( 'coupons' => $coupons, 'args' => $args );
    }

    if ( intval( $args['pagination_per_page'] ) >= sizeof( $coupons ) ) {
        return array( 'coupons' => $coupons, 'args' => $args );
    }

    $coupon_ids = array();

    foreach ( $coupons as $coupon ) {
        $coupon_ids[] = $coupon->id;
    }

    $args['post__in'] = $coupon_ids;

    $args['paginated'] = 1;

    $offset = $args['pagination_per_page'] * ( $args['paginated'] - 1 );

    $coupons = array_slice( $coupons, $offset, $args['pagination_per_page'] );

    // Coupons after paginated array_slice
    $paginated_coupon_ids = array();

    foreach ( $coupons as $coupon ) {
        $paginated_coupon_ids[] = $coupon->id;
    }

    $args['sorted_coupon_ids'] = $paginated_coupon_ids;
    $args['paginated_coupon_ids'] = $paginated_coupon_ids;

    //affcoups_debug_log( '' );
    //affcoups_debug_log( '------- PAGINATION END -------' );
    //affcoups_debug_log( '-----------------------------------------' );

    return array( 'coupons' => $coupons, 'args' => $args );
}
add_filter( 'affcoups_coupons_pagination', 'affcoups_pro_pagination', 10, 2 );

/**
 * Include Pagination Load More button on Page Load
 *
 * @param array $coupons
 * @param array $args
 */
function affcoups_pro_load_more_button( $coupons, $args ) {

    //affcoups_debug_log( '' );
    //affcoups_debug_log( '------- LOAD MORE INIT START -------' );

    if ( affcoups_pro_is_doing_ajax()
      && isset( $_POST['action'] )
      && ( $_POST['action'] == 'affcoups_pro_pagination_action' || $_POST['action'] == 'affcoups_pro_search_filters_action' )
    ){
        return;
    }

    if ( empty( $args['pagination'] )
      || ! isset( $args['post__in'] )
      || ! isset( $args['pagination_per_page'] ) || intval( $args['pagination_per_page'] ) < 1
    ){
        return;
    }

    if ( intval( $args['pagination_per_page'] ) >= sizeof( $args['post__in'] ) ) {
        return;
    }

    $total_coupons = sizeof( $args['post__in'] );

    if ( $total_coupons == '1' ) {
        return;
    }

    $args['total_pages'] = (int) ceil( $total_coupons / ( $args['pagination_per_page'] ) );

    $atts = array(
        'paginated' => $args['paginated'],
        'args' => base64_encode( serialize($args) )
    );

    affcoups_pro_tpl_the_load_more_button( $atts );

    //affcoups_debug_log( '' );
    //affcoups_debug_log( '------- LOAD MORE INIT END -------' );
    //affcoups_debug_log( '-----------------------------------------' );
}
add_action( 'affcoups_after_template', 'affcoups_pro_load_more_button', 10, 2 );

/**
 * Extend style options
 *
 * @param $options
 * @return mixed
 */
function affcoups_pro_style_options( $options ) {

    $options['light'] = __( 'Light', 'affiliate-coupons-pro' );
    $options['dark'] = __( 'Dark', 'affiliate-coupons-pro' );

    return $options;
}
add_filter( 'affcoups_style_options', 'affcoups_pro_style_options', 10 );

/**
 * Extend settings css
 *
 * @param $settings_css
 * @param $prefix
 * @return mixed
 */
function affcoups_pro_settings_css( $settings_css, $prefix ) {

    $options = affcoups_get_options();

    // Clipboard colors
    if ( ! empty( $options['clipboard_bg_color'] ) && ! empty( $options['clipboard_color'] ) ) {
        $settings_css .= $prefix . 'a.affcoups-coupon__button--code-reveal .affcoups-coupon__button-code { background-color: ' . $options['clipboard_bg_color'] . '; color: ' . $options['clipboard_color'] . '; }';
    }

    // Button Colors
    if ( ! empty( $options['button_bg_color'] ) && ! empty( $options['button_color'] ) ) {
        $settings_css .= $prefix . 'a.affcoups-coupon__button--code-reveal .affcoups-coupon__button-text { background-color: ' . $options['button_bg_color'] . '; color: ' . $options['button_color'] . '; }';
        $settings_css .= $prefix . 'a.affcoups-coupon__button--code-reveal .affcoups-coupon__button-text:after { border-left-color: ' . $options['button_bg_color'] . '; }';
    }

    // Highlighted colors
    if ( ! empty( $options['highlighted_border_color'] ) ) {
        $settings_css .= $prefix . '.affcoups-standard .affcoups-coupon--highlighted, ' . $prefix . '.affcoups-coupons-grid .affcoups-coupon--highlighted, ' . $prefix . '.affcoups-widget .affcoups-coupon--highlighted, ' . $prefix . '.affcoups .affcoups-widget-small .affcoups-coupon--highlighted, ' . $prefix . '.affcoups-coupons-list .affcoups-coupon--highlighted { border-color: ' . $options['highlighted_border_color'] . '; }';
    }

    return $settings_css;
}
add_filter( 'affcoups_settings_css', 'affcoups_pro_settings_css', 10, 2 );

/**
 * Maybe extend coupon classes
 *
 * @param $add_classes
 * @param Affcoups_Coupon $Coupon
 * @return mixed
 */
function affcoups_pro_coupon_classes( $add_classes, $Coupon ) {

    $CouponExtended = affcoups_pro_setup_coupon_extended( $Coupon );

    // Featured
    if ( $CouponExtended->is_featured() ) {
        $add_classes[] = 'featured';
    }

    // Highlighted
    if ( $CouponExtended->is_highlighted() ) {
        $add_classes[] = 'highlighted';
    }

    // disable_click_to_reveal
    if ( $CouponExtended->is_click_to_reveal_disable() ) {
        $add_classes[] = 'click_to_reveal_disable';
    }

    return $add_classes;
}
add_filter( 'affcoups_coupon_add_classes', 'affcoups_pro_coupon_classes', 10, 2 );

/**
 * Prepared featured coupon posts
 *
 * @param $coupon_pre_posts
 * @param $args
 * @return mixed
 */
function affcoups_pro_prepare_featured_coupon_posts( $coupon_pre_posts, $args ) {

    if ( isset( $args['affcoups_featured'] ) && false === $args['affcoups_featured'] )
        return $coupon_pre_posts;

    $featured_meta_query = array(
        'key'     => AFFCOUPS_PREFIX . 'coupon_featured',
        'value'   => '1',
        'compare' => '=',
    );

    if ( isset( $args['meta_query'] ) && is_array( $args['meta_query'] ) ) {
        $args['meta_query'][] = $featured_meta_query;
    } else {
        $args['meta_query'] = array( $featured_meta_query );
    }

    $featured_posts_query = new WP_Query( $args );

    if ( isset( $featured_posts_query->posts ) )
        $coupon_pre_posts = $featured_posts_query->posts;

    wp_reset_postdata();

    return $coupon_pre_posts;
}
add_filter( 'affcoups_get_coupons_pre_posts', 'affcoups_pro_prepare_featured_coupon_posts', 10, 2 );

/**
 * Maybe extend get coupons query args
 *
 * @param $args
 * @param $coupon_pre_posts
 * @return mixed
 */
function affcoups_pro_update_coupons_args_after_featured_coupons( $args, $coupon_pre_posts ) {

    // Handle featured coupons
    if ( is_array( $coupon_pre_posts ) && sizeof( $coupon_pre_posts ) > 0 ) {

        $featured_coupon_ids = array();

        foreach( $coupon_pre_posts as $coupon_pre_post ) {

            if ( isset( $coupon_pre_post->ID ) )
                $featured_coupon_ids[] = $coupon_pre_post->ID;
        }

        if ( sizeof( $featured_coupon_ids ) > 0 )
            $args['post__not_in'] = $featured_coupon_ids;
            

        // Handle max return amount
        if ( isset( $args['post__in'] ) && is_array( $args['post__in'] ) ) {
            $args['post__in'] = array_merge(array_diff($args['post__in'], $args['post__not_in']));
            $args['posts_per_page'] = ( intval( $args['posts_per_page'] ) - sizeof( $featured_coupon_ids ) );
        }
    }

    return $args;
}
add_filter( 'affcoups_get_coupons_args', 'affcoups_pro_update_coupons_args_after_featured_coupons', 10, 2 );

/**
 * Maybe extend coupon post type
 *
 * @param $args
 * @return mixed
 */
function affcoups_pro_coupon_post_type( $args ) {

    if ( affcoups_pro_coupon_pages_activated() ) {

        $args_extended = array(
            'public'             => true,
            'publicly_queryable' => true,
            'rewrite'            => array(
                'slug' => affcoups_pro_get_coupon_slug()
            )
        );

        if ( affcoups_pro_coupon_include_in_search_activated() )
            $args_extended['exclude_from_search'] = false;

        // Parse args
        $args = wp_parse_args( $args_extended, $args );
    }

    return $args;
}
add_filter( 'affcoups_coupon_post_type_args', 'affcoups_pro_coupon_post_type' );

/**
 * Maybe extend shortcode default atts
 *
 * @param   array
 * @return  array
 */
function affcoups_pro_shortcodes_default_atts( $default_atts ) {

    // Search Filters
    $default_atts['search_filters'] = null;

    // Pagination
    $default_atts['pagination'] = null;
    $default_atts['pagination_per_page'] = null;

    return $default_atts;
}
add_filter( 'affcoups_shortcode_default_atts', 'affcoups_pro_shortcodes_default_atts' );

/**
 * Maybe extend shortcode args
 *
 * @param $args
 * @param $atts
 * @return mixed
 */
function affcoups_pro_shortcodes_args( $args, $atts ) {

    $options = affcoups_get_options();

    // Coupon detail page
    if ( empty( $args['affcoups_coupon_id'] ) && empty( $args['affcoups_coupon_category'] ) && empty( $args['affcoups_coupon_type'] ) && empty( $args['affcoups_coupon_vendor'] ) && affcoups_pro_is_coupon_page() ) {
        $args['affcoups_coupon_id'] = get_the_ID();
    }

    // Featured
    if ( isset( $atts['featured'] ) && 'false' === $atts['featured'] ) {
        $args['affcoups_featured'] = false;
    }

    // Search Filters
    if ( ! empty( $atts['search_filters'] ) ) {
        $args['search_filters'] = ( 'true' === $atts['search_filters'] ) ? true : false;
    } else {
        $args['search_filters'] = ( isset( $options['search_filters'] ) && '1' === $options['search_filters'] ) ? true : false;
    }

    // Pagination
    if ( ! empty( $atts['pagination'] ) ) {
        $args['pagination'] = ( 'true' === $atts['pagination'] ) ? true : false;
    } else {
        $args['pagination'] = ( isset( $options['pagination'] ) && '1' === $options['pagination'] ) ? true : false;
    }

    // Pagination per page
    if ( ! empty( $args['pagination'] ) ) {

        if ( ! empty( $atts['pagination_per_page'] )
            && is_numeric( $atts['pagination_per_page'] )
            && intval( $atts['pagination_per_page'] ) > 0
        ){
            $args['pagination_per_page'] = intval( $atts['pagination_per_page'] );

        } elseif ( ! empty( $options['pagination_per_page'] )
            && is_numeric( $options['pagination_per_page'] )
            && intval( $options['pagination_per_page'] ) > 0
        ) {
            $args['pagination_per_page'] = intval( $options['pagination_per_page'] );

        } else {
            unset( $args['pagination_per_page'] );
        }
    }

    return $args;
}
add_filter( 'affcoups_shortcode_args', 'affcoups_pro_shortcodes_args', 10, 2 );

/**
 * Maybe redirect coupon url to coupon pages
 *
 * @param $url
 * @param Affcoups_Coupon $Coupon
 * @return mixed
 */
function affcoups_pro_coupon_url_redirect_to_coupon_pages( $url, $Coupon ) {

    if ( affcoups_pro_coupon_redirect_to_page() ) {

        $CouponExtended = affcoups_pro_setup_coupon_extended( $Coupon );

        $url = $CouponExtended->get_page_url();
    }

    return $url;
}
add_filter( 'affcoups_coupon_url', 'affcoups_pro_coupon_url_redirect_to_coupon_pages', 10, 2 );

/**
 * Maybe extend the button args
 *
 * @param $button
 * @param $Coupon
 * @return mixed
 */
function affcoups_pro_coupon_button_args( $button, $Coupon ) {

    // Remove target="_blank" when redirecting to coupon pages
    if ( affcoups_pro_coupon_redirect_to_page() ) {
        $button['target'] = '_self';
    }

    return $button;
}
add_filter( 'affcoups_button_args', 'affcoups_pro_coupon_button_args', 10, 2 );

/**
 * Maybe extend the image args
 *
 * @param $image_args
 * @param $Coupon
 * @return mixed
 */
function affcoups_pro_coupon_the_image_args( $image_args, $Coupon ) {

    // Remove target="_blank" when redirecting to coupon pages
    if ( affcoups_pro_coupon_redirect_to_page() ) {
        $image_args['rel'] = '';
        $image_args['target'] = '_self';
    }

    return $image_args;
}
add_filter( 'affcoups_the_image_args', 'affcoups_pro_coupon_the_image_args', 10, 2 );

/**
 * Maybe hide code when redirecting to coupon pages
 *
 * @param $show_code
 * @param Affcoups_Coupon $Coupon
 * @return bool
 */
function affcoups_pro_hide_code_when_redirect_to_coupon_pages( $show_code, $Coupon ) {

    if ( affcoups_pro_coupon_redirect_to_page() )
        return false;

    return $show_code;
}
add_filter( 'affcoups_show_code', 'affcoups_pro_hide_code_when_redirect_to_coupon_pages', 10, 2 );

/**
 * Maybe extend frontend variables
 *
 * @param $vars
 * @return mixed
 */
function affcoups_pro_frontend_variables( $vars ) {

    $options = affcoups_get_options();

    if ( ! affcoups_pro_stats_disabled() ) {
        $vars['tracking'] = 'true';
    }

    if ( isset( $options['event_tracking'] ) && '1' == $options['event_tracking'] ) {
        $vars['event_tracking'] = 'true';
    }

    return $vars;
}
add_filter( 'affcoups_frontend_vars', 'affcoups_pro_frontend_variables' );

/**
 * Maybe extend coupon container attributes
 *
 * @param $attributes
 * @param $Coupon
 * @return mixed
 */
function affcoups_pro_coupon_container_attributes( $attributes, $Coupon ) {

    if ( affcoups_pro_coupon_redirect_to_page() ) {
        $attributes['redirect-coupon-page'] = 'true';
    }

    return $attributes;
}
add_filter( 'affcoups_coupon_container_attributes', 'affcoups_pro_coupon_container_attributes', 10, 2 );