<?php
/**
 * Search Filters Functions
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Filter Coupon Search Bar
 *
 * @param   array $coupons
 * @param   string $query
 *
 * @return  array
 */
function affcoups_pro_search_bar( $coupons, $query ) {

    if ( sizeof( $coupons ) > 0 ) {

        foreach ( $coupons as $i => $coupon ) {

            // Search within Coupon titles. if empty: Vendor title will be taken
            if ( false !== stripos( $coupon->get_title(), $query ) )
                continue;

            // Search within Coupon descriptions
            if ( false !== stripos( $coupon->get_description(), $query ) )
                continue;

            // Search within Coupon discount codes
            if ( false !== stripos( $coupon->get_code(), $query ) )
                continue;

            unset( $coupons[$i] );
        }
    }

    return $coupons;
}

/**
 * Filter Coupon Taxonomy Dropdowns
 *
 *  AFFCOUPS_COUPON_TYPE_TAXONOMY | AFFCOUPS_COUPON_CATEGORY_TAXONOMY
 *
 * @param   array $coupons
 * @param   string $query
 * @param   string $taxonomy
 *
 * @return  array
 */
function affcoups_pro_search_taxonomy( $coupons, $query, $taxonomy ) {

    if ( sizeof( $coupons ) > 0 ) {

        foreach ( $coupons as $i => $coupon ) {

            $has_taxonomy = false;

            $terms = get_the_terms( $coupon->id, $taxonomy );

            if ( ! $terms || is_wp_error( $terms ) ) {
                unset( $coupons[$i] );
                continue; // for '$coupons' foreach
            }

            foreach ( $terms as $term ) {

                //affcoups_debug_log( $term->term_id );

                if ( $term->term_id == $query ) {
                    $has_taxonomy = true;
                    break; // out of '$terms' foreach
                }
            }

            if ( ! $has_taxonomy ) {
                unset( $coupons[$i] );
            }
        }
    }

    return $coupons;
}

/**
 * Filter Coupon Vendor Dropdown
 *
 * @param   array $coupons
 * @param   string $query
 *
 * @return  array
 */
function affcoups_pro_search_vendor( $coupons, $query ) {

    if ( sizeof( $coupons ) > 0 ) {

        foreach ( $coupons as $i => $coupon ) {

            $vendor_id = get_post_meta( $coupon->id, AFFCOUPS_PREFIX . 'coupon_vendor', true );

            if ( ! $vendor_id || $vendor_id != $query ) {
                unset( $coupons[$i] );
            }
        }
    }

    return $coupons;
}

/**
 * Filter Coupon 'Order By' and 'Order' Dropdowns options
 *
 * @param   array $args
 * @return  array $args
 */
function affcoups_pro_search_ordering( $args ) {

    //affcoups_debug_log( __FUNCTION__ );

    //affcoups_debug_log( '$args BEFORE:' );
    //affcoups_debug_log( $args );

    // Order
    if ( empty( $args['order'] ) ) {

        $args['order'] = 'ASC';
        $args['affcoups_order'] = 'asc';

    } else {
        $order = strtoupper( $args['order'] );

        if ( in_array( $order, [ 'ASC', 'DESC' ] ) ) {

            $args['order'] = $order;
            $args['affcoups_order'] = strtolower( $order );

        } else {
            $args['order'] = 'ASC';
            $args['affcoups_order'] = 'asc';
        }
    }

    // Order by
    if ( empty( $args['orderby'] ) || 'none' === $args['orderby'] ) {

        // "order" property is ignored if "orderby" = "post__in" OR "post_name__in" OR "post_parent__in".
        //  thus we set the default "orderby" value to "name" => "Name (Post)"

        $args['affcoups_orderby'] = 'name';
        $args['orderby'] = 'name';

    } else {
        $orderby = strtolower( $args['orderby'] );

        // 'orderby' (and optional 'meta_key')
        $orderby_options = affcoups_filter_orderby_options( $orderby );

        //affcoups_debug_log( '$orderby_options:' );
        //affcoups_debug_log( $orderby_options );

        if ( ! empty( $orderby_options ) ) {

            $args['orderby'] = $orderby_options['orderby'];
            $args['affcoups_orderby'] = $orderby_options['orderby'];

            if ( isset( $orderby_options['meta_key'] ) ) {
                $args['meta_key'] = $orderby_options['meta_key'];
            }

        } else {
            $args['affcoups_orderby'] = 'name';
            $args['orderby'] = 'name';
        }
    }

    //affcoups_debug_log( '$args AFTER:' );
    //affcoups_debug_log( $args );

    return $args;
}